package middleware

import (
	"compress/flate"
	"compress/gzip"
	"io"
	"net/http"
	"strings"

	"github.com/gin-gonic/gin"
)

// DecompressionMiddleware is a middleware that decompresses the request body
// if the Content-Encoding header is set to gzip or deflate.
func DecompressionMiddleware() gin.HandlerFunc {
	return func(c *gin.Context) {
		// Check Content-Encoding header
		encoding := c.Request.Header.Get("Content-Encoding")

		if encoding == "" {
			c.Next()
			return
		}

		var reader io.ReadCloser
		var err error

		// Decompress the request body
		switch {
		case strings.Contains(encoding, "gzip"):
			reader, err = gzip.NewReader(c.Request.Body)
			if err != nil {
				c.AbortWithError(http.StatusBadRequest, err)
				return
			}
			defer reader.Close()
			c.Request.Body = io.NopCloser(reader)

		case strings.Contains(encoding, "deflate"):
			reader = flate.NewReader(c.Request.Body)
			defer reader.Close()
			c.Request.Body = io.NopCloser(reader)
		}

		// Continue processing other requests
		c.Next()
	}
}
